let

// a type of equality proofs

type Eq <a> <b> = <f> -> f <a> -> f <b>

function refl <a> : Eq <a> <a> =
  fun <f> (x : f <a>) => x

function symm <a> <b> (x : Eq <a> <b>) : Eq <b> <a> =
  fun <f> => x <fun <c> => f <c> -> f <a>> (refl <a> <f>)

function tran <a> <b> <c> (x : Eq <a> <b>) (y : Eq <b> <c>) : Eq <a> <c> =
  fun <f> (p : f <a>) => y <f> (x <f> p)

function from <a> <b> (x : Eq <a> <b>) : a -> b =
  x <fun <x> => x>

function to <a> <b> (x : Eq <a> <b>) : b -> a =
  from (symm x)

// a simple encoded GADT with equality proofs

data Term <a> =
  TNat  (Eq <a> <Nat>, Nat)
| TBool (Eq <a> <Bool>, Bool)
| TPlus (Eq <a> <Nat>, Term <Nat>, Term <Nat>)
| TAnd  (Eq <a> <Bool>, Term <Bool>, Term <Bool>)
| TEq   (Eq <a> <Bool>, Term <Nat>, Term <Nat>)
| TIf   (Term <Bool>, Term <a>, Term <a>)

// smart constructors

val tNat  = fun n => TNat  (refl <Nat>, n)
val tBool = fun b => TBool (refl <Bool>, b)
val tPlus = fun x => TPlus (refl <Nat>, x.1, x.2)
val tAnd  = fun x => TAnd  (refl <Bool>, x.1, x.2)
val tEq   = fun x => TEq   (refl <Bool>, x.1, x.2)
val tIf   = fun x => TIf   x

// a typed evaluator

function eval <a> (t : Term <a>) : a =
  case t of
    TNat  (p, n) => to p n
  | TBool (p, b) => to p b
  | TPlus (p, x, y) => to p (eval x + eval y) 
  | TAnd  (p, x, y) => to p (eval x && eval y)
  | TEq   (p, x, y) => to p (eval x == eval y)
  | TIf   (c, t, e) => if eval c then eval t else eval e
  end

val test = tIf (tEq (tNat 2, tPlus (tNat 1, tNat 1)), 
                tAnd (tBool true, tBool true),
                tBool false)

in

eval test

end
