// bplc Account.bpl
let

open System.List

type Account =
  class
    method deposit  : Nat -> ()
    method withdraw : Nat -> ()
    method balance  : Nat
  end

exception Insufficient Nat

val standard-account =
  class
    local
      val bal = ref 0
    in
      method deposit (d : Nat) : () = bal := !bal + d
      method withdraw (d : Nat) : () =
        if d ># !bal then throw (Insufficient d)
                     else bal := !bal - d
      method balance : Nat = !bal
    end
  end

data Transaction = In Nat | Out Nat

function total (list : List <Transaction>) : Nat =
  case list of
    Nil => 0
  | Cons ( In  d, list' ) => total list' + d
  | Cons ( Out d, list' ) => total list' - d
  end

val account-with-log =
  class
    local
      val log = ref Nil
    in
      method deposit (d : Nat) : () =
        log := Cons ( In d, !log )
      method withdraw (d : Nat) : () =
        if d ># self.balance then throw (Insufficient d)
                             else log := Cons ( Out d, !log )
      method balance : Nat = total (!log)
      method log : Ref <List <Transaction>> = log
    end
  end

val account-with-clearable-log =
  class
    inherit account-with-log
    method clear : () =
      let
        val s = total (!self.log)
      in
        self.log := Cons ( In s, Nil )
      end
  end

// val myaccount = new standard-account // : Account
// val myaccount = new account-with-log
val myaccount = new account-with-clearable-log

in
  try  
    myaccount.deposit (200);
    myaccount.withdraw (50);
    myaccount.withdraw (60);
    myaccount.deposit (40);
    myaccount.withdraw (60);
    write ("OK.\n")
  catch Insufficient d =>
    write ("Insufficient funds!\n");
    write ("  Can't withdraw "); write (show (d)); write ("$.\n")
  end;
  System.IO.println (!myaccount.log);
  myaccount.clear;
  System.IO.println (!myaccount.log)
end
