// requires the Source type
// self: open recursion

// `self' refers to the object, on which the method was called.

// Efficient version:
// Requires bounded-recursion; uses references, back-patching
// and the covariant `Source' type.

let

type Set-Counter =
  ( get = () -> Nat, set = Nat -> (), inc = () -> () )

type Counter-Rep = ( x = Ref <Nat> )

/*

// erst `self', dann `rep, using recursion

function set-counter-class
  <r> // eigentlich r <: Counter-Rep
  (self : r -> Set-Counter) : Set-Counter =
  ( get    = \ rep () => !rep.x,
    set    = \ rep i  => rep.x := i,
    inc    = \ rep () => self.set rep (self.get rep () + 1) )

val new-set-counter =
  let
    val method-table = rec obj => set-counter-class <Counter-Rep> obj
  in
    \ () => 
      let val rep = ( x = ref 0 ) in
        ( get = method-table.get rep,
          set = method-table.set rep,
          inc = method-table.inc rep )
      end
  end
*/

// erst `self', dann `rep, using back-patching

exception Null

// mit back-patching
function set-counter-class
  <r> // eigentlich r <: Counter-Rep
  (self : Source <r -> Set-Counter>)
  (rep  : r) : Set-Counter =
  ( get    = \ () => !rep.x,
    set    = \ i  => rep.x := i,
    inc    = \ () => ((!self) rep).set (((!self) rep).get () + 1) )

val new-set-counter =
  let
  // class table wird einmal statisch allokiert
    val obj  = ref (\ rep => throw Null)
    val obj' = set-counter-class <Counter-Rep> obj
  in
  obj := obj'; // back-patching
  \ () => obj' ( x = ref 0 )
  end

// ohne back-patching
function set-counter-class
  <r> // eigentlich r <: Counter-Rep
  (self : Source <r -> Set-Counter>)
  (rep  : r) : Set-Counter =
  ( get    = \ () => !rep.x,
    set    = \ i  => rep.x := i,
    inc    = \ () => (self rep).set ((self rep).get () + 1) )

val new-set-counter =
  let
    val obj = rec obj => set-counter-class <Counter-Rep> obj
  in
  \ () => obj ( x = ref 0 )
  end

in
let
  val c1 = new-set-counter ()
  val c2 = new-set-counter ()
in
  write (show (c1.inc ()));
  write (show (c2.get ()));
  write (show (c1.inc (); c2.inc (); c1.get ()));
  write (show (c2.get ()))
end
end
