// will not pass the type checker: mismatch between object and class types
// requires Stack and Assoc
// bplc -n -l Stack.bpl -l Assoc.bpl Calculator.bpl

let

// model

type Register =
  class
    method getValue : Nat
    method setValue : Nat -> ()
  end

val register =
  class
    local
      val v = ref 0
    in
      method getValue : Nat = !v
      method setValue (n : Nat) : () = v := n
    end
  end

//

open Stack

type Calculator =
  class
    method getStack : Stack <Nat>
    method getRegister : Register
  end

val calculator =
  class
    local
      val stack = new (stack <Nat>)
      val reg   = new register
    in
      method getStack : Stack <Nat> = stack
      method getRegister : Register = reg
    end
  end

//

type Operator =
  class
    method operate : ()
  end

function digit-op (calc : Calculator, d : Nat) : Operator =
  class
    method operate : () =
      calc.getRegister.setValue (calc.getRegister.getValue * 10 + d)
  end

function enter-op (calc : Calculator) : Operator =
  class
    method operate : () =
      calc.getStack.push (calc.getRegister.getValue);
      calc.getRegister.setValue 0
  end

function equals-op (calc : Calculator) : Operator =
  class
    method operate : () =
      calc.getRegister.setValue (calc.getStack.pop)
  end

function bin-op (calc : Calculator, op : (Nat, Nat) -> Nat) : Operator =
  class
    method operate : () =
      let
        val x = calc.getStack.pop
        val y = calc.getStack.pop
        val z = op (y, x)
      in
        calc.getStack.push z
      end
  end

// tui

module StringEq : EQ <String> = struct function eq? (x,y) = x == y end
module Dict = Assoc <String, Operator> StringEq

val calculatorApp =
  class
    local
      val calc = new calculator
      val dict =
        Dict.insert ("?", new 
          class
            method operate : () =
              System.IO.writeln "Welcome to the calculator.";
              System.IO.writeln "Digits augment the contents of the register.";
              System.IO.writeln "Enter (`;') puts the register on the stack.";
              System.IO.writeln "Operators (`+',`-',`*',`/') operate on the stack.";
              System.IO.writeln "Equals (`=') pops the stack to the register."
          end,
        Dict.insert ("0", new (digit-op (calc, 0)),
        Dict.insert ("1", new (digit-op (calc, 1)),
        Dict.insert ("2", new (digit-op (calc, 2)),
        Dict.insert ("3", new (digit-op (calc, 3)),
        Dict.insert ("4", new (digit-op (calc, 4)),
        Dict.insert ("5", new (digit-op (calc, 5)),
        Dict.insert ("6", new (digit-op (calc, 6)),
        Dict.insert ("7", new (digit-op (calc, 7)),
        Dict.insert ("8", new (digit-op (calc, 8)),
        Dict.insert ("9", new (digit-op (calc, 9)),
        Dict.insert (";", new (enter-op (calc)),
        Dict.insert ("=", new (equals-op (calc)),
        Dict.insert ("+", new (bin-op (calc, \ (i, j) => i + j)),
        Dict.insert ("-", new (bin-op (calc, \ (i, j) => i - j)),
        Dict.insert ("*", new (bin-op (calc, \ (i, j) => i * j)),
        Dict.insert ("/", new (bin-op (calc, \ (i, j) => i / j)),
        Dict.empty)))))))))))))))))
    in
      method run : () =
        (Dict.lookup ("?", dict)).operate;
        while true do
          try
            System.IO.println (calc.getRegister.getValue);
            write "digit/;/op/= : ";
            foreach <String> (System.IO.readln ()) (\ c => (Dict.lookup (c, dict)).operate)
          catch
            Top | Pop => System.IO.writeln "Stack is empty."
          | Lookup c  => System.IO.writeln ("Character `" + c + "' not recognized.")
          end
        end
    end
  end

val tui = new calculatorApp

in
tui.run
end
