///////////////////////////////////////////////////////////////////////////////
// lazy evaluation
///////////////////////////////////////////////////////////////////////////////

type Thunk <a> = () -> a

val force = value

local
  open Maybe
in
  function lazy <a> (comp : Thunk <a>) : Thunk <a> =
    let val box = ref Nothing in
      { case !box of 
          Nothing => let val v = force comp in
                       box := Just v; v
                     end
        | Just v  => v
        end }
  end
end

// with black-holing

data State <a> = Empty | Transit | Value a

exception Black-hole

function lazy <a> (comp : Thunk <a>) : Thunk <a> =
  let val box = ref Empty in
    { case !box of 
        Empty   => let val v = box := Transit; force comp in
                     box := Value v; v
                   end
      | Transit => throw Black-hole
      | Value v => v
      end }
  end

// rec n => n + 1                        // exception Loop
// force (rec n => {force n + 1})        // loops
// force (rec n => lazy {force n + 1})   // black-holes
// force (lazy (rec n => {force n + 1})) // loops

///////////////////////////////////////////////////////////////////////////////
// Streams
///////////////////////////////////////////////////////////////////////////////

data State <a> = Cons (a, Thunk <State <a>>)

type Stream <a> = Thunk <State <a>>

function head <a> (s : Stream <a>) : a =
  case force s of Cons (x, xs) => x end

function tail <a> (s : Stream <a>) : Stream <a> =
  case force s of Cons (x, xs) => xs end

function show-stream <a> (n : Nat, s : Stream <a>) : () =
  if n == 0 then
    ()
  else 
    put-line (show (head s));
    show-stream <a> (n - 1, tail s)

/*
Typische Idiome zur Konstruktion von Strmen:
  lazy { Cons (_, _) }
  rec s => lazy { Cons (_, _ s _) }
  rec s => lazy { Cons (_, _ {force s} _) }
*/

val ones = rec s => lazy { Cons (1, s) }

val undef = rec s => lazy { Cons (1, tail s) } // black-holes
// val undef = rec s => lazy { Cons (1, { force (tail s) } ) } // loops

function const (n : Nat) : Stream <Nat> =
  rec s => lazy { Cons (n, s) }

function from (n : Nat) : Stream <Nat> =
  lazy { Cons (n, from (n + 1)) }

// show-stream (1000, ones)
// show-stream (1000, const 4711)
// show-stream (1000, from 0)

//
// carry = 0 \/ carry + 1
//

function merge <a> (s1 : Stream <a>, s2 : Stream <a>) : Stream <a> =
  lazy { Cons (head s1, merge (s2, tail s1)) }

function add (s1 : Stream <Nat>, s2 : Stream <Nat>) : Stream <Nat> =
  lazy { Cons (head s1 + head s2, add (tail s1, tail s2)) }

val carry =
  rec s => merge (const 0, add ({force s}, const 1)) // ok
//  rec s => merge (const 0, add (s, const 1)) // loops
//  rec s => lazy { Cons (0, merge (add (s, const 1), const 0)) } // ok
//  rec s => lazy { force (merge (const 0, add (s, const 1))) } // ok

// show-stream (1000, carry)

//
// prime numbers
//

function candidates (n : Nat) : Stream <Nat> =
  lazy { Cons (n, candidates (n + 2)) }

function filter <a> (p : a -> Bool, s : Stream <a>) : Stream <a> =
  lazy { if p (head s) then Cons (head s, filter <a> (p, tail s))
                       else force (filter <a> (p, tail s)) }
// !! der Test `p (head s)' darf nicht auerhalb der lazy {} Klammern stehen !!

function prime-wrt (s : Stream <Nat>) (n : Nat) : Bool =
  let val p = head s in
    p * p ># n || n % p /= 0 && prime-wrt (tail s) n
  end

val primes = rec ps => lazy { Cons (2, filter (prime-wrt ps, candidates 3)) }

// show-stream (1000, primes)

//
// Fibonacci using Memotables
//

val fibonacci =
  rec fib => 
    array [100] i =>
      lazy { if i =< 1 then 1 
                       else force fib.[i - 1] + force fib.[i - 2] }

// force fibonacci.[99]

// Pascal's triangle

val pascal =
  rec a =>
    array [30] n =>
      array [n] i =>
        lazy { if i == 0 || i == n - 1 then
                 1
               else
                 force a.[n - 1].[i - 1] + force a.[n - 1].[i] }

// let val p = pascal.[7] in array [size p] i => force p.[i] end
// array [size pascal] n => array [size pascal.[n]] i => force pascal.[n].[i]
