val ?    = "Type :? for help"
val help = ? 

val put-string = write

function put-line (s : String) : () =
  put-string s; put-string "\n"

function get-line () : String =
  let 
    val c = get-char () 
  in
    if c == "\n" then ""
                 else c ^ get-line ()
  end

local
  val chatty = ref false
in
  function log (message : String) : () =
    if !chatty then put-string message else ()
end

data Ordering = LT | EQ | GT

exception Error String

function error (message : String) : Void =
  throw (Error message)

function not (b : Bool) : Bool =
  if b then false else true

function value <a> (block : () -> a) : a =
  block ()

function id <a> (x : a) : a =
  x

module String =
struct
  val () = log "Loading <String> ..\n"

  // re-exporting built-in functions
  val null? = null?
  val head  = head
  val tail  = tail
  val chr   = chr
  val ord   = ord
  val show  = show

  function compare (s : String, t : String) : Ordering =
    if      s #< t then    LT
    else if s == t then    EQ
    else /* s ># t then */ GT

  function digit? (c : String) : Bool =
    ord "0" =< ord c && ord c =< ord "9"

  function lower? (c : String) : Bool =
    ord "a" =< ord c && ord c =< ord "z"

  function space? (c : String) : Bool =
    c == " " || c == "\t" || c == "\n" || c == "\v" || c == "\f" || c == "\r"

  function upper? (c : String) : Bool =
    ord "A" =< ord c && ord c =< ord "Z"

  function alpha? (c : String) : Bool =
    upper? c || lower? c

  function alnum? (c : String) : Bool =
    alpha? c || digit? c

  function length (s : String) : Nat =
    if null? s then 0
               else 1 + length (tail s)

  function reverse (s : String) : String =
    if null? s then ""
               else reverse (tail s) ^ head s
end // String

module Bool =
struct
  val () = log "Loading <Bool> ..\n"

  // re-exporting built-in functions
  val true  = true
  val false = false
  val not   = not

  function compare (a : Bool, b : Bool) : Ordering =
    if a then if b then EQ else GT
         else if b then LT else EQ
end // Bool

module Nat =
struct
  val () = log "Loading <Nat> ..\n"

  function compare (a : Nat, b : Nat) : Ordering =
    if      a #< b then    LT
    else if a == b then    EQ
    else /* a ># b then */ GT

  function minimum (a : Nat, b : Nat) : Nat =
    if a =< b then a else b

  function maximum (a : Nat, b : Nat) : Nat =
    if a >= b then a else b

  function power (x : Nat, n : Nat) : Nat =
    if      n == 0     then 1
    else if n % 2 == 0 then     power (x * x, n / 2)
                       else x * power (x * x, n / 2)

  function read (s : String) : Nat =
    let 
      function nat (s : String) : Nat =
        if null? s then 0
                   else ord (head s) - ord "0" + 10 * nat (tail s)
    in nat (String.reverse s) end
end // Nat

local
// pseudo-random number generator (linear congruential method, constants from Numerical Recipes in C)
  val a = 1664525
  val c = 1013904223
  val m = Nat.power(2, 32)
  val x = ref 271965
in
  function random () : Nat =
    x := (a * !x + c) % m; !x
end

module Maybe =
struct
  val () = log "Loading <Maybe> ..\n"

  data Maybe <a> = Nothing | Just a

  function compare <a> (cmp : (a, a) -> Ordering) (x : Maybe <a>, y : Maybe <a>) : Ordering =
    case (x, y) of (Nothing, Nothing) => EQ
                |  (Nothing, Just _)  => LT
                |  (Just _, Nothing)  => GT
                |  (Just a, Just b)   => cmp (a, b)
                end
  function map <a, b> (f : a -> b) (maybe : Maybe <a>) : Maybe <b> =
    case maybe of Nothing => Nothing
               |  Just a  => Just (f a)
               end
end // Maybe

module List =
struct
  val () = log "Loading <List> ..\n"

  data List <a> = Nil | Cons (a, List <a>)

  function compare <a> (cmp : (a, a) -> Ordering) (x : List <a>, y : List <a>) : Ordering =
    case (x, y) of (Nil, Nil)     => EQ
                |  (Nil, Cons _)  => LT
                |  (Cons _, Nil)  => GT
                |  (Cons (a, as), Cons (b, bs)) => 
                     case cmp (a, b) of LT => LT
                                     |  EQ => compare <a> cmp (as, bs)
                                     |  GT => GT
                                     end
                end

  function map <a, b> (f : a -> b) (list : List <a>) : List <b> =
    case list of Nil          => Nil
              |  Cons (a, as) => Cons (f a, map <a, b> f as)
              end

  function foreach <a> (list : List <a>) (action : a -> ()) : () =
    case list of Nil          => ()
              |  Cons (x, xs) => action x; foreach <a> xs action
              end

  function print <a> (p : a -> ()) (list : List <a>) : () =
    case list of Nil          => ()
              |  Cons (x, xs) => p x; print p xs
              end

  function fold <a, b> (nil : b, cons : (a, b) -> b) : List <a> -> b =
    rec fun recurse (list : List <a>) : b =>
      case list of Nil          => nil
                |  Cons (x, xs) => cons (x, recurse xs)
                end

  function append <a> (list1 : List <a>, list2 : List <a>) : List <a> =
    case list1 of Nil          => list2
               |  Cons (n, ns) => Cons (n, append <a> (ns, list2))
               end

  function reverse-append <a> (list1 : List <a>, list2 : List <a>) : List <a> =
    case list1 of Nil          => list2
               |  Cons (n, ns) => reverse-append <a> (ns, Cons (n, list2))
               end

  function reverse <a> (list : List <a>) : List <a> = reverse-append <a> (list, Nil)

  function length <a> (list : List <a>) : Nat =
    case list  of  Nil          =>  0
               |   Cons (x, xs) =>  1 + length <a> xs
               end

  function split <a : Type> (n : Nat, list : List <a>) : (List <a>, List <a>) =
    if n == 0  then
      (Nil, list)
    else
      case list of
        Nil          => (Nil, Nil)
      | Cons (x, xs) => let val (xs1, xs2) = split (n - 1, xs)
                        in  (Cons (x, xs1), xs2) end
      end

  function split-while <a> (pred : a -> Bool, list : List <a>) : (List <a>, List <a>) =
    case list of Nil          => (Nil, Nil)
              |  Cons (x, xs) => if pred x then
                                   let val (xs1, xs2) = split-while <a> (pred, xs)
                                   in  (Cons (x, xs1), xs2) end
                                 else
                                   (Nil, list)
              end
           
  function between (lower : Nat, upper : Nat) : List <Nat> =
    if lower ># upper then Nil
                      else Cons (lower, between (lower + 1, upper))

  // bottom-up mergesort
  function merge-by <a> (less-equal : (a, a) -> Bool) : (List <a>, List <a>) -> List <a> =
    let
      function merge (list1 : List <a>, list2 : List <a>) : List <a> =
        case (list1, list2) of
          (Nil, xs) 
        | (xs, Nil)                        => xs
        | (Cons (x1, xs1), Cons (x2, xs2)) =>
            if less-equal (x1, x2) then Cons (x1, merge (xs1, list2))
                                   else Cons (x2, merge (list1, xs2))
        end
    in 
      merge
    end

  function sort-by <a> (less-equal : (a, a) -> Bool) : List <a> -> List <a> =
    let
      function runs (list : List <a>) : List <List <a>> =
        case list of Nil => Nil
                  |  Cons (x, Nil) => Cons (Cons (x, Nil), Nil)
                  |  Cons (x1, Cons (x2, xs)) =>
                       if less-equal (x1, x2) then Cons (Cons (x1, Cons (x2, Nil)), runs xs)
                                              else Cons (Cons (x2, Cons (x1, Nil)), runs xs)
                  end

      function merge-pairwise (lists : List <List <a>>) : List <List <a>> =
        case lists of Nil                         => Nil
                   |  Cons (xs, Nil)              => Cons (xs, Nil)
                   |  Cons (xs1, Cons (xs2, xss)) => Cons (merge-by <a> less-equal (xs1, xs2), merge-pairwise xss)
                   end

      function merge-all (lists : List <List <a>>) : List <a> =
        case lists of Nil            => Nil
                   |  Cons (xs, Nil) => xs
                   |  _              => merge-all (merge-pairwise lists)
                   end
    in 
      fun list => merge-all (runs list)
    end

  function explode (s : String) : List <String> =
    if null? s then Nil
               else Cons (head s, explode (tail s))

  function implode (list : List <String>) : String =
    case list of Nil            => ""
              |  Cons (s, rest) => s ^ implode rest
              end

  function from-array <a> (arr : Array <a>) : List <a> =
    let function listify (i : Nat) : List <a> =
      if i >= size arr then Nil else Cons (arr.[i], listify (i + 1))
    in
      listify 0
    end

end // List

module Cont =
struct
  val () = log "Loading <Cont> ..\n"

  exception Null

  // construction of a ref cell of type `Ref (Cont a)'
  function alloc <a> : Ref <Cont <a>> =
    letcc q : Cont <Ref <Cont <a>>> in
      letcc k : Cont <a> in continue q (ref k) end ;
      throw Null
    end
end // Cont

module System =
struct
  val () = log "Loading <System> ..\n"

  // re-exporting built-in and top-level functions
  val size   = size
  val error  = error
  val random = random
  val log    = log

  module Bool = Bool

  module Nat = Nat

  module Maybe = Maybe

  module List = List

  module String = String

  module Cont = Cont

  module Control =
  struct
    val () = log "Loading <Control> ..\n"

    // for loop, finite repetition
    function for (lower : Nat, upper : Nat) (body : Nat -> ()) : () =
      if lower ># upper then ()
                        else body lower; 
                             for (lower + 1, upper) body

    // endless repetition
    function forever (body : () -> ()) : Void =
      body (); 
      forever body
  end // Control

  module IO =
  struct
    val () = log "Loading <IO> ..\n"

    local
      open List // to bring List, Nil, Cons into Scope
    in
      // re-exporting built-in functions
      val get-char       = get-char
      val put-char       = put-char
      val read-line      = read-line
      val write          = write
      val read-from-file = read-from-file
      val write-to-file  = write-to-file

      function readln () : List <String> =
        let 
          val c = get-char () 
        in
          if c == "\n" then Nil
                       else Cons ( c, readln ())
        end
  
      function writeln (s : String) : () =
        write s; write "\n"

      function print (x : Value) : () =
        write (show x)

      function println (x : Value) : () =
        write (show x); write "\n"
      // `print' and `println' are pseudo-polymorphic (like `show')
    end //local

  end // IO

end // System