module Int =
struct

type Int = (pos = Nat, neg = Nat)

// Normalisierung

function normalize (i : Int) : Int =
    (pos = i.pos - i.neg, neg = i.neg - i.pos)

// Klassifikation

function negative? (i : Int) : Bool =
  i.pos #< i.neg

function zero? (i : Int) : Bool =
  i.pos == i.neg

function positive? (i : Int) : Bool =
  i.pos ># i.neg

// Konversion, Betrag und Vorzeichen

function int (n : Nat) : Int =
  (pos = n, neg = 0)

function neg (n : Nat) : Int =
  (pos = 0, neg = n)

function pos (n : Nat) : Int =
  (pos = n, neg = 0)

function abs (i : Int) : Nat =
  (i.pos - i.neg) + (i.neg - i.pos)

function signum (i : Int) : Int =
  case Nat.compare (i.pos, i.neg) of
    LT => (pos = 0, neg = 1)
  | EQ => (pos = 0, neg = 0)
  | GT => (pos = 1, neg = 0)
  end

// arithmetische Operationen

function negate (i : Int) : Int =
  (pos = i.neg, neg = i.pos)

function add (i : Int, j : Int) : Int =
  normalize (pos = i.pos + j.pos, neg = i.neg + j.neg)

function sub (i : Int, j : Int) : Int =
  normalize (pos = i.pos + j.neg, neg = i.neg + j.pos)

function mul (i : Int, j : Int) : Int =
  normalize (pos = i.pos * j.pos + i.neg * j.neg,
             neg = i.pos * j.neg + i.neg * j.pos )

// fr div und mod gilt: a = add (mul (div (a, b), b), mod (a, b))

function div (i : Int, j : Int) : Int =
  if i.pos >= i.neg then
    if j.pos >= j.neg then
      (pos = (i.pos - i.neg) / (j.pos - j.neg), neg = 0)
    else
      let val b = j.neg - j.pos in (pos = 0, neg = (i.pos - i.neg + b - 1) / b) end
  else
    if j.pos >= j.neg then
      let val b = j.pos - j.neg in (pos = 0, neg = (i.neg - i.pos + b - 1) / b) end
    else
      (pos = (i.neg - i.pos) / (j.neg - j.pos), neg = 0)

function mod (i : Int, j : Int) : Int =
  if i.pos >= i.neg then
    if j.pos >= j.neg then
      (pos = (i.pos - i.neg) % (j.pos - j.neg), neg = 0)
    else
      let val b = j.neg - j.pos
          val r = (i.pos - i.neg) % b
      in  if r == 0 then (pos = 0, neg = 0) else (pos = 0, neg = b - r) end
  else
    if j.pos >= j.neg then
      let val b = j.pos - j.neg 
          val r = (i.neg - i.pos) % b
      in  if r == 0 then (pos = 0, neg = 0) else (pos = b - r, neg = 0) end
    else
      (pos = 0, neg = (i.neg - i.pos) % (j.neg - j.pos))

// Vergleichsoperationen

function compare (i : Int, j : Int) : Ordering =
  Nat.compare (i.pos + j.neg, i.neg + j.pos)

function less (i : Int, j : Int) : Bool =
  i.pos + j.neg #< i.neg + j.pos

function less-equal (i : Int, j : Int) : Bool =
  i.pos + j.neg =< i.neg + j.pos

function equal (i : Int, j : Int) : Bool =
  i.pos + j.neg == i.neg + j.pos

function not-equal (i : Int, j : Int) : Bool =
  i.pos + j.neg /= i.neg + j.pos

function greater-equal (i : Int, j : Int) : Bool =
  i.pos + j.neg >= i.neg + j.pos

function greater (i : Int, j : Int) : Bool =
  i.pos + j.neg ># i.neg + j.pos

end // Int
